' Pentomino Rectangle Packing Puzzle
' Rev 1.0.0 William M Leue 25-Jan-2022
' Rev 1.1.0 5-Feb-2022 Added save/restore of the puzzle state.
'   and added the 5th puzzle type.
option default integer
option base 1

const MSIZE  = 5
const NMINOS = 12
const NROTS  = 4
const NFLIPS = 2
const NALPHA = 26
const NSIZES = 5
const CSIZE  = 30
const SSIZE  = 20
const ISIZE  = 15
const STORE  = 0
const BOARD  = 1
const DROP   = 2
const NSVAL  = 4
const MAXVX  = 12
const NMINL  = 5

const MXSAVE = 66
const YSTART = 75
const XOFF   = 60

const ESC   = 27
const FIVE  = 53
const SLASH = 47
const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const SPACE = 32
const HOME  = 134
const ATSGN = 64
const QUEST = 63
const F1    = 145
const F2    = 146
const ENTER = 13

const CMDPOP_X = 50
const CMDPOP_Y = 100
const CMDPOP_W = 700
const CMDPOP_H = 340

const SRPOP_X = 50
const SRPOP_Y = 50
const SRPOP_W = 700
const SRPOP_H = 500
const SAVE_PUZ = 1
const LOAD_PUZ = 2

const F     = 70
const I     = 73
const L     = 76
const N     = 78
const P     = 80
const T     = 84
const U     = 85
const V     = 86
const W     = 87
const X     = 88
const Y     = 89
const Z     = 90

' Globals
dim mindex(NALPHA)
dim minos(MSIZE, MSIZE, NMINOS)
dim dminos(MSIZE, MSIZE, NMINOS)
dim colors(NMINOS)
dim mnames$(NMINOS)
dim store_locs(2, NMINOS)
dim intro_locs(2, NMINOS)
dim minolocs(NMINL, NMINOS)
dim sclocs(2, NSIZES)
dim nvertices(NMINOS)
dim vertices(2, MAXVX, NMINOS)
dim dvertices(2, MAXVX, NMINOS)
dim solutions(NSVAL, NMINOS, NSIZES)

dim cx = MM.HRES\2 
dim cy = 420
dim rwidth = 12
dim rheight = 5
dim cells(rwidth, rheight)
dim dcells(rwidth, rheight)
dim bx = 0
dim by = 0
dim cursor_row = rheight\2
dim cursor_col = rwidth\2 - 1
dim selected_mino = 0
dim rchoice = 1
dim sample = 0

' Main Program
'open "debug.txt" for output as #1
cls
ReadMinoData
ReadSolutions
InitCursor
do
  ShowIntro
  InitPuzzle
  DrawPuzzle
  HandleEvents
loop
end

' Read the index files, pentomino matrices, and colors
sub ReadMinoData
  local col, row, rot, flip, mino, alpha, i, j, v
  for mino = 1 to NMINOS
    for row = 1 to MSIZE
      for col = 1 to MSIZE
        read minos(col, row, mino)
        dminos(col, row, mino) = minos(col, row, mino)
      next col
    next row
    read nvertices(mino)
    for v = 1 to nvertices(mino)
      for c = 1 to 2
        read vertices(c, v, mino)
        dvertices(c, v, mino) = vertices(c, v, mino)
      next c
    next v
  next mino
  for alpha = 1 to NALPHA
    read mindex(alpha)
  next alpha
  for mino = 1 to NMINOS
    read colors(mino)
  next mino
  for mino = 1 to NMINOS
    read mnames$(mino)
  next mino
  for mino = 1 to NMINOS
    read store_locs(1, mino)
    read store_locs(2, mino)
  next mino
  for mino = 1 to NMINOS
    read intro_locs(1, mino)
    read intro_locs(2, mino)
  next mino
  for i = 1 to NSIZES
    for j = 1 to 2
      read sclocs(j, i)
    next j
  next i
end sub

' Read the sample solutions for each rectangle size
sub ReadSolutions
  local mino, index, size
  for size = 1 to NSIZES
    for mino = 1 to NMINOS
      for index = 1 to NSVAL
        read solutions(index, mino, size)
      next index
    next mino
  next size
end sub

' Initialize the cursor
sub InitCursor
  gui cursor on 1, 0, 0, rgb(red)
  gui cursor show
  UpdateCursor
end sub

' Initialize a new Puzzle
sub InitPuzzle
  local mino, row, col, i, c, v
  select case rchoice
    case 1 : rwidth = 10 : rheight = 6
    case 2 : rwidth = 12 : rheight = 5
    case 3 : rwidth = 15 : rheight = 4
    case 4 : rwidth = 20 : rheight = 3
    case 5 : rwidth = 8  : rheight = 8
  end select
  bx = cx - (rwidth\2)*CSIZE - (CSIZE\2)*(rwidth mod 2)
  by = cy - (rheight\2)*CSIZE - (CSIZE\2)*(rheight mod 2)
  erase cells, dcells
  dim cells(rwidth, rheight)
  dim dcells(rwidth, rheight)
  cursor_col = sclocs(1, rchoice)
  cursor_row = sclocs(2, rchoice)
  UpdateCursor
  for mino = 1 to NMINOS
    for i = 1 to NMINL
      minolocs(i, mino) = STORE
    next i
    minolocs(4, mino) = 1
    minolocs(5, mino) = 1
    for row = 1 to MSIZE
      for col = 1 to MSIZE
        dminos(col, row, mino) = minos(col, row, mino)
      next col
    next row
  next mino
  for row = 1 to rheight
    for col = 1 to rwidth
      if rchoice = 5 then
        if (row = 4 or row = 5) and (col = 4 or col = 5) then
          cells(col, row) = -2
        end if
      else
        cells(col, row) = 0
        dcells(col, row) = 0
      end if
    next col
  next row
  for mino = 1 to NMINOS
    for v = 1 to nvertices(mino)
      for c = 1 to 2
        dvertices(c, v, mino) = vertices(c, v, mino)
      next c
    next v
  next mino
  selected_mino = 0
end sub

' Handle keyboard events
sub HandleEvents
  local z$
  local cmd, k, move, prev_selected, pflag
  do
    z$ = INKEY$
    do
      z$ = INKEY$
    loop until z$ <> ""
    pflag = 0
    cmd = asc(UCASE$(z$))
    move = 0
    select case cmd
      case ESC
        cls
        SavePuzzle
        gui cursor hide
        gui cursor off
        end
      case HOME
        exit do
      case ATSGN
        sample = 1
        ShowSampleSolution
      case QUEST
        PrintCommands
      case F1
        SavePuzzle "./SavedPuzzles"
      case F2
        ListSavedPuzzleFiles
      case asc("A") to asc("Z")
        if sample = 1 then continue do
        if rchoice = 5 then
          if (cursor_row - 4 or cursor_row = 5) and (cursor_col = 4 or cursor_col = 5) then
            continue do
          end if
        end if
        prev_selected = 0
        if selected_mino > 0 then
          prev_selected = selected_mino
          TogglePentomino selected_mino, STORE
        end if
        k = mindex(cmd - asc("A") + 1)
        if k > 0 and k <> prev_selected then
          if BoardHasPentomino(k) then
            RemovePentomino k
            TogglePentomino k, STORE
          else
            TogglePentomino k, BOARD
          end if
          prev_selected = 0
        end if        
        DrawBoard
        DrawStore
      case FIVE
        if sample = 1 then continue do
        RotatePentomino selected_mino
        DrawBoard
      case SLASH
        if sample = 1 then continue do
        FlipPentomino selected_mino
        DrawBoard
      case SPACE
        if sample = 1 then continue do
        if selected_mino > 0 then pflag = 1
        PickDropPentomino
        DrawBoard
      case UP to RIGHT
        move = 1
      case else
    end select
    UpdateCursor cmd
    if selected_mino > 0 then DrawBoard
    DrawStore
  loop
end sub

' Move the cursor around on board cells
' If there is an active piece, it follows the cursor.
sub UpdateCursor cmd
  local x, y
  select case cmd
    case UP
      if cursor_row > 1 then
        cursor_row = cursor_row-1
      end if
    case DOWN
      if cursor_row < rheight then
        cursor_row = cursor_row+1
      end if
    case LEFT
      if cursor_col > 1 then
        cursor_col = cursor_col-1
      end if
    case RIGHT
      if cursor_col < rwidth then
        cursor_col = cursor_col+1
      end if
    case else
  end select
  x = cx + (cursor_col-rwidth\2)*CSIZE - CSIZE\2 - (CSIZE\2)*(rwidth mod 2)
  y = cy + (cursor_row-rheight\2)*CSIZE - CSIZE\2 - (CSIZE\2)*(rheight mod 2)
  gui cursor x, y
  if selected_mino > 0 then
    PlacePentomino selected_mino
  end if    
end sub

' Toggle a Pentomino between Store and Board
' Pieces taken out of the store go to selected status
' and are entered into the dcells matrix.
' (Note 'which' is call by reference from selected_mino
' so changing selected_mino also changes 'which', so that
' needs to be done last for each case.)
sub TogglePentomino which, newStatus
  local v
  minolocs(1, which) = newStatus
  v = minolocs(1, which)  
  if v = STORE then
    minolocs(2, which) = 0
    minolocs(3, which) = 0
    minolocs(4, which) = 1
    minolocs(5, which) = 1
    ClearDCells
    selected_mino = 0
  else
    minolocs(1, which) = BOARD
    minolocs(2, which) = cursor_col
    minolocs(3, which) = cursor_row 
    selected_mino = which
  end if
end sub

' Place the selected pentomino on the board.
' The board placement is actually just to the dcell matrix, not the
' cell matrix. 
' Pentominos can extend past the board edges and overlap
' other pieces until the 'drop' command is given, at which time  no
' overlaps are allowed. Note that pieces that extended outside the
' board edge will be clipped to just the board rectangle when rendered.
sub PlacePentomino which
  local v, row, col, mrow, mcol, tcol, trow, ok
  ClearDCells
  for row = 1 to MSIZE
    brow = row + cursor_row - 3
    if brow < 1 or brow > rheight then continue for
    for col = 1 to MSIZE
      if dminos(col, row, which) = 1 then
        bcol = col + cursor_col - 3
        if bcol < 1 or bcol > rwidth then continue for
        dcells(bcol, brow) = which
      end if
    next col
  next row
end sub

' Clear the dcells matrix. This happens each time the cursor moves
' while dragging an active piece, or when a piece is dropped into
' position.
sub ClearDCells
  local row, col
  for row = 1 to rheight
    for col = 1 to rwidth
      dcells(col, row) = 0
    next col
  next row
end sub

' Pick up or Drop a Pentomino. To 'drop' means to
' copy the piece from the dcells() array to the cells()
' array, clear the dcells() array, and turn off the
' active piece. To 'pick' means to remove the piece under
' the cursor from the cells array, put it back into the
' dcells() array, and make it the active piece.
sub PickDropPentomino
  local row, col, brow, bcol, v, index
  if selected_mino > 0 then
    if not Interference(selected_mino) then
      minolocs(1, selected_mino) = DROP
      minolocs(2, selected_mino) = cursor_col
      minolocs(3, selected_mino) = cursor_row
      for row = 1 to MSIZE
        brow =  row+cursor_row-3
        for col = 1 to MSIZE
          v = dminos(col, row, selected_mino)
          if v > 0 then
            bcol = col+cursor_col-3
            cells(bcol, brow) = selected_mino
            dcells(bcol, brow) = 0
          end if
        next col
      next row
      selected_mino = 0
      if IsSolved() then
        if sample = 1 then
          text mm.hres\2, 599, "(Sample Solution)", "CB", 5,, rgb(yellow)
        else
          text mm.hres\2, 599, "Puzzle Solved!", "CB", 5,, rgb(green)
        end if
      end if
    end if
  else
    index = cells(cursor_col, cursor_row)
    if index > 0 then
      cursor_col = minolocs(2, index)
      cursor_row = minolocs(3, index)
      UpdateCursor 0
      RemovePentomino index
      selected_mino = index
    end if
  end if
end sub

' Remove a previously fixed pentomino from the cells array
sub RemovePentomino which
  local row, col
  for row = 1 to rheight
    for col = 1 to rwidth
      if cells(col, row) = which then
        cells(col, row) = 0
      end if
    next col
  next row
  minolocs(1, which) = BOARD
end sub

' Return 1 if the cells() array has cells with value 'which'
function BoardHasPentomino(which)
  local row, col
  for row = 1 to rheight
    for col = 1 to rwidth
      if cells(col, row) = which then
        BoardHasPentomino = 1
        exit function
      end if
    next col
  next row
  BoardHasPentomino = 0
end function

' Return 1 if the proposed location for a pentomino drop
' is off the board or overlaps with an already-placed
' pentomino. No side effects!
function Interference(which)
 local row, col, brow, bcol, v
  for row = 1 to MSIZE
    brow =  row+cursor_row-3
    for col = 1 to MSIZE
      v = dminos(col, row, which)
      if v > 0 then
        bcol = col+cursor_col-3
        if bcol < 1 or bcol > rwidth or brow < 1 or brow > rheight then
          Interference = 1
          exit function
        end if
        if cells(bcol, brow) <> 0 then
          Interference = 1
          exit function
        end if
      end if
    next col
  next row
  Interference = 0
end function

' Draw the Puzzle
sub DrawPuzzle
  cls
  DrawBoard
  DrawStore
end sub

' Draw the Board
' Draw empty cells first, then dropped pieces,
' and finally any active piece.
sub DrawBoard
  local row, col, x, y, w, h, c, v, e, i, mino
  w = rwidth*CSIZE
  h = rheight*CSIZE
  for row = 1 to rheight
    y = cy - h\2 + (row-1)*CSIZE
    for col = 1 to rwidth
      x = cx - w\2 + (col-1)*CSIZE
      e = rgb(black)
      if rchoice = 5 then
        if (row = 4 or row = 5) and (col = 4 or col = 5) then
          c = rgb(black)
        else
          c = rgb(white)    
        end if
      else
        c = rgb(white)
      end if
      box x, y, CSIZE, CSIZE,, e, c
    next col
  next row
  for i = 1 to NMINOS
    if minolocs(1, i) = DROP then
      DrawPolygon i
    end if
  next i
  for row = 1 to rheight
    y = cy - h\2 + (row-1)*CSIZE
    for col = 1 to rwidth
      x = cx - w\2 + (col-1)*CSIZE
      mino = dcells(col, row)
      if selected_mino > 0 and mino = selected_mino then
        e = rgb(white)
        c = colors(mino)
        box x, y, CSIZE, CSIZE,, e, c
      end if
    next col
  next row
end sub

' Draw the Store of Pentominos around the puzzle grid
sub DrawStore
  local mino, x, y
  for mino = 1 to NMINOS
    x = store_locs(1, mino)
    y = store_locs(2, mino)
    if minolocs(1, mino) = STORE then
      DrawStoreMino mino, SSIZE, x, y
      text x+MSIZE*SSIZE\2, y+MSIZE*SSIZE, mnames$(mino), "CT", 4
    else
      box x, y, MSIZE*SSIZE, MSIZE*SSIZE,, rgb(black), rgb(black)
    end if
  next mino
end sub

' Draw a pentomino in the store
sub DrawStoreMino which, size, x, y
  local row, col, c, uc, v, mx, my
  c = colors(which)
  for row = 1 to MSIZE
    my = y + (row-1)*size
    for col = 1 to MSIZE
      mx = x + (col-1)*size
      v = minos(col, row, which)
      if v = 1 then box mx, my, size, size,, rgb(black), c
    next col
  next row
end sub         

' Draw the polygon for the current pentomino by scaling and offseting the vertices
' in the current dvertices() array for that pentomino. The polygon location is
' determined by the minolocs values for that pentomino, with a 3-cell offset because
' of the different origins for polygon versus cell-based definitions.
sub DrawPolygon which
  local v, x, y, xv, yv, bx, by
  local uxv(MAXVX), uyv(MAXVX)
  bx = cx - (rwidth\2)*CSIZE - (CSIZE\2)*(rwidth mod 2)
  by = cy - (rheight\2)*CSIZE - (CSIZE\2)*(rheight mod 2)
  for v = 1 to nvertices(which)
    xv = minolocs(2, which) + dvertices(1, v, which)
    yv = minolocs(3, which) + dvertices(2, v, which)
    uxv(v) = bx + xv*CSIZE - 3*CSIZE
    uyv(v) = by + yv*CSIZE - 3*CSIZE
  next v
  gui cursor hide
  polygon nvertices(which), uxv(), uyv(), rgb(black), colors(which)
  gui cursor show
end sub

' Rotate a selected pentomino
sub RotatePentomino which
  local row, col, rrow, rcol
  local v, xv, vy, rxv, ryv
  local temp(MSIZE, MSIZE)
  if which = 0 then exit sub
  for row = 1 to MSIZE
    for col = 1 to MSIZE
      rrow = col
      rcol = MSIZE - row + 1
      temp(rcol, rrow) = dminos(col, row, which)
      dminos(col, row, which) = 0
    next col
  next row
  for row = 1 to MSIZE
    for col = 1 to MSIZE
      dminos(col, row, which) = temp(col, row)
    next col
  next row
  for v = 1 to nvertices(which)
    xv = dvertices(1, v, which)
    yv = dvertices(2, v, which)
    ryv = xv
    rxv = MSIZE - yv
    dvertices(1, v, which) = rxv
    dvertices(2, v, which) = ryv
  next v
  inc minolocs(4, which)
  if minolocs(4, which) > NROTS then minolocs(4, which) = 1
end sub

' Do a 3D horizontal flip of a selected pentomino
sub FlipPentomino which
  local row, col, rcol
  local v xv, rxv
  local temp(MSIZE, MSIZE)
  if which = 0 then exit sub
  for row = 1 to MSIZE
    for col = 1 to MSIZE
      rcol = MSIZE - col + 1
      temp(rcol, row) = dminos(col, row, which)
      dminos(col, row, which) = 0
    next col
  next row
  for row = 1 to MSIZE
    for col = 1 to MSIZE
      dminos(col, row, which) = temp(col, row)
    next col
  next row
  for v = 1 to nvertices(which)
    xv = dvertices(1, v, which)
    rxv = MSIZE - xv
    dvertices(1, v, which) = rxv
  next v
  minolocs(5, which) = 3 - minolocs(5, which)
end sub

' Function to detect a solved puzzle (1 = solved)
function IsSolved()
  local row, col
  for row = 1 to rheight
    for col = 1 to rwidth
      if cells(col, row) = 0 then
        IsSolved = 0
        exit function
      end if
    next col
  next row
  IsSolved = 1
end function

' Introduction and Rules
sub ShowIntro
  local z$
  local mino
  cls
  sample = 0
  gui cursor hide
  print "Pentominos are Polyominos of order 5; that is, they are shapes made by connecting"
  print "5 squares at their edges. There are 12 unique pentominos, or 18 if flips in the"
  print "third dimension are intepreted to be unique. Here are the 12 unique shapes that can"
  print "be made from 5 connected squares:"
  for mino = 1 to NMINOS
    DrawStoreMino mino, ISIZE, intro_locs(1, mino), intro_locs(2, mino)
    text intro_locs(1, mino)+3*ISIZE, intro_locs(2, mino)-2, mnames$(mino), "CB", 7
  next mino
    
  text 0, 230, ""
  print "Half of these pentominos have no symmetries and therefore can be flipped over in the"
  print "third dimension to produce an orientation that cannot be reached by just doing 2D"
  print "rotations: the F, L, N, P, W, and Z pentominos. The other 6 have at least one axis"
  print "of symmetry and can produce any orientation by just 2D rotations."
  print ""
  print "The goal of this puzzle is to fit all 12 pentominos into a 60-unit rectangle so that"
  print "there are no gaps or parts outside the rectangle. There are 4 different rectangles"
  print "that have 60 squares and which will fit all the pentominos: 10x6, 12x5, 15x4, and 20x3."
  print "You will choose one of these rectangles and try to fit the pieces into it.
  print ""
  print "You can move a piece from the 'store' above the board by typing the letter of the"
  print "pentomino. (These letters are chosen because the pentomino more or less resembles the"
  print "shape of that letter.) Once the piece is on the board, use the arrow keys to move the"
  print "piece around to get it into your chosen position. Type the '5' key on the keypad to"
  print "rotate the piece 90 degrees. You can repeat this command any number of times. Type the"
  print "'/' key on the keypad to flip the piece in 3D. You can also repeat this command."
  print ""
  print "Once you have positioned the piece where you want it and with the orientation you want,"
  print "press the spacebar to 'drop' the piece onto the board. The piece will now remain fixed"
  print "in position regardless of where the cursor is. If you later decide to move the piece"
  print "that you placed, move the cursor until it is over the piece and press the spacebar"
  print "again. The piece will be picked up so that you can move it, rotate it, and flip it."
  print "The 'drop' command will not work if the active piece overlaps any other piece or if any"
  print "of the active piece's squares lie outside of the board edges."
  print ""
  print "Continue adding pieces to the board until all 12 pentominos are on the board with no gaps"
  print "or overlaps. The game will let you know when the puzzle has been solved."
  print ""
  do
    print "Choose rectangle size 10x6, 12x5, 15x4, 20x3, or 8x8 with a 2x2 hole (1-5): ";
    input "", z$
    rchoice = val(z$)
    if rchoice >= 1 and rchoice <= NSIZES then exit do
  loop
  gui cursor show
  PrintCommands
end sub  

' Print a pop-up window with the commands
sub PrintCommands
  local z$
  local x, y
  box CMDPOP_X, CMDPOP_Y, CMDPOP_W, CMDPOP_H,, rgb(white), rgb(black)
  x = CMDPOP_X+10
  y = CMDPOP_Y+20
  text x+CMDPOP_W\2, y, "Pentomino Puzzle Commands", "CT", 4,, rgb(green), -1
  y = CMDPOP_Y+60
  print @(x, y) "A-Z:   (The pentomino's letter F, I, L, N, P, T, U, V, W, X, Y, or Z)"
  inc y, 15
  print @(x, y) "          Choose a pentomino to work with or return it to the store."
  inc y, 15
  print @(x, y) "5:      Rotate the currently active pentomino"
  inc y, 15
  print @(x, y) "/:      Flip the currently active pentomino in 3D"
  inc y, 15
  print @(x, y) "SPACE:  Drop the currently active pentomino onto the board or pick it up again."
  inc y, 15
  print @(x, y) "HOME:   Restart the puzzle"
  inc y, 15
  print @(x, y) "@:      Show a sample solution for the chosen board size"
  inc y, 15
  print @(x, y) "F1:     Save the current puzzle state to a file"
  inc y, 15
  print @(x, y) "F2:     Load a puzzle from a file"
  inc y, 15
  print @(x, y) "?:      Show this pop-up window again."
  inc y, 15
  print @(x, y) "ESC:    Quit the Game"
  inc y, 30
  print @(x, y) "Besides the sample solution, there are other solutions for each board size:
  inc y, 15
  print @(x, y) " 6x10: 2339 solutions 5x12: 1010 solutions 4x15: 368 solutions 3x20: 2 solutions"
  inc y, 30
  print @(x, y) "Note that NumLock must be OFF for the keypad keys to work."
  inc y, 40
  print @(x, y) "Press any key to close this window."
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  DrawPuzzle
end sub

' show a sample solution for the current board size
sub ShowSampleSolution
  local mino, rot, flip, r
  InitPuzzle
  ClearDCells
  for mino = 1 to NMINOS    
    cursor_col = solutions(1, mino, rchoice)
    cursor_row = solutions(2, mino, rchoice)
    selected_mino = mino
    PlacePentomino mino
    rot  = solutions(3, mino, rchoice)
    flip = solutions(4, mino, rchoice)
    for r = 1 to rot-1
      RotatePentomino mino
    next r
    if flip = 2 then
      FlipPentomino mino
    end if
    PickDropPentomino
  next mino
  ClearDCells
  DrawBoard
end sub    

' Save the current puzzle state to a file
' Filename: S_P<puzzle type>_<date>_<time>_(Partial or Solution).psv
' Time has been modified with '-' instead of ':' to make it compatible.
sub SavePuzzle todir$
  local mino, i
  local stat$, t$, d$, tm$, c$, wd$
  wd$ = cwd$
  chdir todir$
  if IsSolved() then stat$ = "Solved" else stat$ = "Partial"
  t$ = time$
  d$ = date$
  tm$ = ""
  for i = 1 to len(t$)
    c$ = mid$(t$, i, 1)
    if c$ = ":" then
      cat tm$, "-"
    else
      cat tm$, c$
    end if
  next i
  local f$ = "S_P" + str$(rchoice) + "_" + d$ + "_" + tm$ + "_" + stat$ + ".psv"
  on error skip 1
  open f$ for output as #2
  if MM.ERRNO <> 0 then
    print "Error opening file '";f$;"' for saving puzzle"
    exit sub
  end if
  print #2, rchoice
  for mino = 1 to NMINOS
    if BoardHasPentomino(mino) then
      print #2, minolocs(2, mino);",";minolocs(3, mino);",";
      print #2, minolocs(4,mino);",";minolocs(5,mino)
    else
      print #2, "0,0,0,0"
    end if
  next mino
  close #2
  chdir wd$
end sub

' Load a previous puzzle state from a file
sub LoadPuzzle f$
  local mino, par, i, rot, flip
  local buf$
  on error skip 1
  open f$ for input as #2
  if MM.ERRNO <> 0 then
    cls
    print "Error opening file '";f$;"' for reading solution"
    end
  end if
  line input #2, buf$
  rchoice = val(buf$)
  InitPuzzle
  for mino = 1 to NMINOS
    line input #2, buf$
    if buf$ = "0,0,0,0" then continue for
    cat buf$, ","
    selected_mino = mino
    for i = 1 to 4
      par = val(field$(buf$, i, ","))
      select case i
        case 1 : cursor_col = par
        case 2 : cursor_row = par
        case 3 : rot = par
        case 4 : flip = par
      end select
    next i
    PlacePentomino mino
    for r = 1 to rot-1
      RotatePentomino mino
    next r
    if flip = 2 then
      FlipPentomino mino
    end if
    PickDropPentomino
  next mino
  close #2
  ClearDCells
  DrawPuzzle
end sub

' Get a list of files from a directory
' Arguments are directory, file pattern, the maximum number of names to list,
' the number of matching files found, and the list of filenames returned.
sub GetFileList d$, f$, maxnames, numfiles, results$()
  local buf$, wd$
  local n
  wd$ = cwd$
  chdir d$
  n = 0
  buf$ = dir$(f$, FILE)
  do while n <= maxnames and buf$ <> ""
    inc n
    results$(n) = buf$
    buf$ = dir$()
  loop
  numfiles = n
  chdir wd$
end sub

' List the saved puzzle files and let the user choose one for loading
sub ListSavedPuzzleFiles
  local results$(MXSAVE)
  local d$ = "./SavedPuzzles"
  local f$ = "*.psv"
  local t$
  local numfiles = 0
  local i, x1, x2, y
  gui cursor hide
  GetFileList d$, f$, MXSAVE, numfiles, results$()
  cls
  box 10, 70, 780, 520
  line mm.hres\2, 50, mm.hres\2, 569
  text mm.hres\2, 20, "Saved Puzzle Files", "CT", 4,, rgb(green)
  text mm.hres\2, 38, "Use Arrow Keys to move to file, press Enter to choose", "CT"
  text mm.hres\2, 53, "Filenames: puzzle size, date, time, and Partial or Solved", "CT"
  x1 = XOFF
  x2 = mm.hres\2+XOFF
  y = YSTART
  for i = 1 to MXSAVE\2
    t$ = str$(i) + ": " + results$(i)
    text x1, y, t$
    inc y, 15
  next i
  y = YSTART
  for i = MXSAVE\2+1 to MXSAVE
    t$ = str$(i) + ": " + results$(i)
    text x2, y, t$
    inc y, 15
  next i
  HandleChooseFileEvents numfiles, results$()
end sub

' Event handler for choosing a saved puzzle file
sub HandleChooseFileEvents numfiles, results$()
  local z$
  local cmd, chrow, chcol, index, hit
  chrow = 1
  chcol = 1
  hit = 0
  DrawPointer chrow, chcol
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case ESC
        gui cursor show
        exit do
      case UP
        if chrow > 1 then chrow = chrow-1
      case DOWN
        if chrow < MXSAVE\2 then chrow = chrow+1
      case LEFT
        chcol = 1
      case RIGHT
        chcol = 2
      case ENTER
        index = (chcol-1)*MXSAVE\2 + chrow
        if len(results$(index)) > 4 then
          hit = 1
          exit do
    end select
    DrawPointer chrow, chcol
  loop
  if hit = 1 then
    LoadPuzzle "./SavedPuzzles/" + results$(index)
  end if
end sub

' Draw the pointer for the file chooser
sub DrawPointer row, col
  local x, y
  local xv(3), yv(3)
  static prow = -1
  static pcol = -1
  if prow > 0 then
    if pcol = 1 then x = XOFF-15 else x = mm.hres\2+XOFF-15
    y = YSTART+7 + (prow-1)*15
    xv(1) = x : yv(1) = y
    xv(2) = x-10 : yv(2) = y-8
    xv(3) = x-10 : yv(3) = y+8
    polygon 3, xv(), yv(), rgb(black), rgb(black)
  end if
  if col = 1 then x = XOFF-15 else x = mm.hres\2+XOFF-15
  y = YSTART+7 + (row-1)*15
  xv(1) = x : yv(1) = y
  xv(2) = x-10 : yv(2) = y-8
  xv(3) = x-10 : yv(3) = y+8
  polygon 3, xv(), yv(), rgb(red), rgb(red)
  prow = row : pcol = col
end sub

' Read-only database for Pentominos. First row for each
' pentomino is for the presence (1) or absence (0) of a square.
' Second row is the polyon vertex equivalent for the outside
' edges of the pentomino. The transformation rules for polygons
' are similar to those for squares under rotation and reflection.
' (the vertices need to be scaled and offset by the cell size
' to turn them into x,y coordinates.)
' The vertex rows start with a vertex count.

' F Pentomino
data 0,0,0,0,0,  0,0,1,1,0,  0,1,1,0,0,  0,0,1,0,0,  0,0,0,0,0
data 10, 2, 1, 4, 1, 4, 2, 3, 2, 3, 4, 2, 4, 2, 3, 1, 3, 1, 2, 2, 2

' I pentomino
data 0,0,1,0,0,  0,0,1,0,0,  0,0,1,0,0,  0,0,1,0,0,  0,0,1,0,0
data 4, 2, 0, 3, 0, 3, 5, 2, 5

' L pentomino
data 0,0,1,0,0,  0,0,1,0,0,  0,0,1,0,0,  0,0,1,1,0,  0,0,0,0,0
data 6, 2, 0, 3, 0, 3, 3, 4, 3, 4, 4, 2, 4

' N penomino
data 0,0,1,0,0,  0,0,1,0,0,  0,1,1,0,0,  0,1,0,0,0,  0,0,0,0,0
data 8, 2, 0, 3, 0, 3, 3, 2, 3, 2, 4, 1, 4, 1, 2, 2, 2

' P pentomino
data 0,0,0,0,0,  0,1,1,0,0,  0,1,1,0,0,  0,1,0,0,0,  0,0,0,0,0
data 6, 1, 1, 3, 1, 3, 3, 2, 3, 2, 4, 1, 4

' T pentomino
data 0,0,0,0,0,  0,1,1,1,0,  0,0,1,0,0,  0,0,1,0,0,  0,0,0,0,0
data 8, 1, 1, 4, 1, 4, 2, 3, 2, 3, 4, 2, 4, 2, 2, 1, 2 

' U pentomino
data 0,0,0,0,0,  0,1,0,1,0,  0,1,1,1,0,  0,0,0,0,0,  0,0,0,0,0
data 8, 1, 1, 2, 1, 2, 2, 3, 2, 3, 1, 4, 1, 4, 3, 1, 3

' V pentomino
data 0,0,1,0,0,  0,0,1,0,0,  0,0,1,1,1,  0,0,0,0,0,  0,0,0,0,0
data 6, 2, 0, 3, 0, 3, 2, 5, 2, 5, 3, 2, 3

' W pentomino
data 0,0,0,0,0,  0,1,1,0,0,  0,0,1,1,0,  0,0,0,1,0,  0,0,0,0,0
data 10, 1, 1, 3, 1, 3, 2, 4, 2, 4, 4, 3, 4, 3, 3, 2, 3, 2, 2, 1, 2

' X pentomino
data 0,0,0,0,0,  0,0,1,0,0,  0,1,1,1,0,  0,0,1,0,0,  0,0,0,0,0
data 12, 2, 1, 3, 1, 3, 2, 4, 2, 4, 3, 3, 3, 3, 4, 2, 4, 2, 3, 1, 3, 1, 2, 2, 2

' Y pentomino
data 0,0,1,0,0,  0,0,1,0,0,  0,1,1,0,0,  0,0,1,0,0,  0,0,0,0,0
data 8, 2, 0, 3, 0, 3, 4, 2, 4, 2, 3, 1, 3, 1, 2, 2, 2

' Z pentomino
data 0,0,0,0,0,  0,1,1,0,0,  0,0,1,0,0,  0,0,1,1,0,  0,0,0,0,0
data 8, 1, 1, 3, 1, 3, 3, 4, 3, 4, 4, 2, 4, 2, 2, 1, 2

' Pentomino Lookup based on letter name
data 0, 0, 0, 0, 0, 1, 0, 0, 2, 0, 0, 3, 0
data 4, 0, 5, 0, 0, 0, 6, 7, 8, 9, 10, 11, 12

' Pentomino colors
data rgb(200, 190, 120), rgb(219, 117, 171), rgb(136, 171, 105), rgb(0, 204, 102)
data rgb(0, 153, 0),     rgb(0, 153, 153),   rgb(51, 204, 251),  rgb(0, 112, 192)
data rgb(0, 0, 255),     rgb(204, 0, 153),   rgb(255, 0, 0),     rgb(255, 153, 0)

' Pentomimo Names
data "F", "I", "L", "N", "P", "T", "U", "V", "W", "X", "Y", "Z"

' Store locations
data  90,   10,   190,  10,  290,  10,  390,  10,  490,  10,  590,  10
data  90,  150,   190, 150,  290, 150,  390, 150,  490, 150,  590, 150

' Into page locations
data  90,   70,   190,  70,  290,  70,  390,  70,  490,  70,  590,  70
data  90,  155,   190, 155,  290, 155,  390, 155,  490, 155,  590, 155

' Starting cursor locations
data 5, 3, 6, 3, 7, 2, 10, 2, 1, 1

' Solved puzzle data for 6x10
' Each row: col, row, rotation, flip
data  6, 4, 2, 1 'F
data  1, 3, 1, 1 'I
data  8, 2, 2, 2 'L
data  6, 6, 2, 1 'N
data  3, 2, 1, 1 'P
data  2, 5, 3, 1 'T
data 10, 5, 4, 1 'U
data 10, 1, 3, 1 'V
data  4, 5, 3, 1 'W
data  4, 3, 1, 1 'X
data  5, 1, 4, 2 'Y
data  8, 4, 1, 1 'Z

' Solved puzzle for 5x12
data  9, 4, 1, 1 'F
data  3, 5, 2, 1 'I
data  1, 3, 1, 1 'L
data  2, 3, 1, 2 'N
data  4, 2, 2, 2 'P
data  7, 4, 3, 1 'T
data  5, 4, 1, 1 'U
data 12, 5, 4, 1 'V
data  6, 2, 4, 1 'W
data  8, 2, 1, 1 'X
data 10, 1, 4, 2 'Y
data 11, 3, 1, 2 'Z

' Solved puzzle for 4x15
data 12, 2, 2, 1 'F
data  9, 1, 2, 1 'I
data  1, 2, 3, 2 'L
data  6, 1, 4, 1 'N
data  6, 3, 3, 2 'P
data 14, 3, 3, 1 'T
data  3, 4, 1, 1 'U
data 15, 1, 3, 1 'V
data  8, 3, 4, 1 'W
data  3, 2, 1, 1 'X
data  9, 4, 2, 1 'Y
data 11, 3, 2, 2 'Z

' Solved puzzle for 3x20
data 11, 2, 2, 1 'F
data  6, 3, 2, 1 'I
data  8, 1, 2, 1 'L
data  9, 3, 2, 1 'N
data  5, 2, 2, 1 'P
data 13, 2, 1, 1 'T
data  1, 2, 2, 1 'U
data 20, 1, 3, 1 'V
data 15, 2, 4, 1 'W
data  3, 2, 1, 1 'X
data 16, 3, 2, 1 'Y
data 18, 2, 2, 2 'Z

' Solved puzzle for 8x8 with 2x2 hole in center
data 4, 3, 2, 2
data 6, 8, 2, 1
data 1, 7, 1, 1
data 2, 7, 1, 2
data 7, 3, 4, 1
data 7, 2, 2, 1
data 8, 6, 4, 1
data 1, 1, 2, 1
data 4, 6, 3, 1
data 6, 6, 1, 1
data 5, 1, 4, 2
data 2, 3, 3, 2

' End of Source Code
